// Objekt zur Steuerung der amira DAC98 Karte
// 12.11.1997 B.Osterkamp
// Dies ist die Funktionsdeklaration

#include <math.h>
#include <conio.h>
#include <dos.h>

#ifdef _Windows
#include "wtools.h"
#endif


#include "drv\DAC98.h"

DAC98::DAC98( int adress )
{
	Base	= 	adress;	// Basisadresse der DAC98

	WR_DATA =	0x0004;	// Datenregister schreiben
	RD_DATA =	0x0004;	// Datenregister auslesen
	Clock 	=	2.0e6;    // Clock fr Timer und intern Counter in MHz
	CounterGate=	0;
	CounterJMP=	0;
	aout0	=	0;
	aout1	=	0;
	intr	=	0;



	// Variablen der Digitalen Kanle
	output_status_DAC98	= 0;
	input_status_DAC98	= 0;
	intr_status		= 0x2f;

	ddm_adr[0]=DDM02a_CS;
	ddm_adr[1]=DDM02b_CS;
	ddm_adr[2]=DDM02c_CS;

	Init();
};

DAC98::~DAC98()
{
	Exit();
};


int DAC98::Identifikation( void )
{
	outp( Base , Iden_CS );
	return( inp(Base+RD_DATA)==DAC98ID ? 1 : 0 );
}

void DAC98::Init ( void )
{


	if (!Identifikation()) return;

	// Programmierung des IO-Bausteins
	outp(Base,IO_CS+0x0003);
	outp(Base+WR_DATA,0x008A);

	// Alle dig. Ports auf 0
	output_status_DAC98 = 0x0000;
	// Digitale Kanle 0-7 (PortA 0-7)
	outp(Base,IO_CS+0x0000);
	outp(Base+WR_DATA,(unsigned char)output_status_DAC98);

	// Digitale Kanle 8-11 (PortC 0-3)
	outp(Base,IO_CS+0x0002);
	outp(Base+WR_DATA,(unsigned char)(output_status_DAC98>>8));

	// Status der dig. Eingaenge lesen
	ReadDigitalInputs();

	// Analogen Ausgnge auf 0Volt Setzen
	WriteAnalogVolt(0,0);
	WriteAnalogVolt(1,0);

	// DDM's initialisieren
	ResetAllDDM();
	ReadAllDDM();

	// Interrupts und Clocksignal setzen (0x2f)
	outp(Base,Interrupt_CS);
	outp(Base+WR_DATA,intr_status);



	// Timer initialisieren
	GateTimer( 0 );
	GateCounter( 0 );
	SetTimer( 0xffffffffL );
	SetCounter( 0xffff );

	TestCounterJMP();
}

int DAC98::Exit ( void )
{
	ResetAllDDM();
	WriteAllDigital(0);
	WriteAnalogVolt(0,0);
	WriteAnalogVolt(1,0);
	GateCounter(0);
	GateTimer(0);
	return 1;
}

int DAC98::Setup (void)
{
	// Suche der Basisadresse
	for( Base = 0x0300; Base < 0x0380; Base+=0x0010 )
	{
	  if( Identifikation() ) return 1;
	}
	return 0;
}

void DAC98::SetClock( int mode )
{
	switch (mode)
	{
		case Clk8MHz   : Clock=8.0e6;break;
		case Clk4MHz   : Clock=4.0e6;break;
		case Clk2MHz   : Clock=2.0e6;break;
		case Clk1MHz   : Clock=1.0e6;break;
		case Clk500kHz : Clock=0.5e6;break;
		case Clk250kHz : Clock=0.250e6;break;
		case Clk125kHz : Clock=0.125e6;break;
		case Clk62kHz  : Clock=0.0625e6;break;
		default	       : break;
	}

	intr_status = ( (  mode << 4 ) & 0xF0 ) | ( intr_status & 0x0F );

	outp(Base,Interrupt_CS);

	outp(Base+WR_DATA,intr_status);
};


void DAC98::WriteDigital (int channel, int value)
{
	if( channel<0 ) return;
	if( channel<12 ){

	  output_status_DAC98 = value ?
			 (output_status_DAC98 | (0x0001<<channel)) :
			 (output_status_DAC98 &~(0x0001<<channel));

	  // Digitale Kanle 0-7 (PortA 0-7)
	  outp(Base,IO_CS+0x0000);
	  outp(Base+WR_DATA,(unsigned char)output_status_DAC98);

	  // Digitale Kanle 8-11 (PortC 0-3)
	  outp(Base,IO_CS+0x0002);
	  outp(Base+WR_DATA,(unsigned char)(output_status_DAC98>>8));

	}
}

void DAC98::WriteAllDigital( int value )
{
	output_status_DAC98 &= 0xFF00;
	output_status_DAC98 |= (0x00FF & value );

	// Digitale Kanle 0-7 (PortA 0-7)
	outp(Base,IO_CS+0x0000);
	outp(Base+WR_DATA,(unsigned char)output_status_DAC98);

	// Digitale Kanle 8-11 (PortC 0-3)
	outp(Base,IO_CS+0x0002);
	outp(Base+WR_DATA,(unsigned char)(output_status_DAC98>>8));

}


unsigned int DAC98::ReadDigitalInputs( void )
{
	unsigned int wert = 0;

	outp(Base,IO_CS+0x0001);
	wert  = ( unsigned int )inp( Base+RD_DATA );

	outp(Base,IO_CS+0x0002);
	wert |= ( unsigned int )inp( Base+RD_DATA ) << 8;

	return wert;
}

int DAC98::ReadDigital( int channel )
{
	unsigned int mask;

	if( channel<0 ) return 0;
	if( channel<12 ){
	  if( channel>7 ) channel += 4;
	  mask = 0x0001<<channel;
	  return( ( ReadDigitalInputs() & mask ) ? 1 : 0 );
	}
	return 0;
}

void DAC98::SetCounter(unsigned int count)
{

	timer_counter2=count;
	outp( Base , Timer_CS+3 );
	outp( Base+WR_DATA, 0xb6 ); 
	outp( Base , Timer_CS+2 );
	outp( Base+WR_DATA, ( unsigned char)count );
	outp( Base+WR_DATA, ( unsigned char)(count>>8) );
}

unsigned int DAC98::GetCounter( void )
{
	unsigned int wert=0;

	outp( Base, Timer_CS+3 );
	outp( Base+WR_DATA, 0xe8); 

	outp( Base, Timer_CS+2 );

	if ( inp( Base+RD_DATA ) & 64 ) wert |= timer_counter2;
	else
	{
		outp( Base, Timer_CS+3 );
		outp( Base+WR_DATA, 0xD8);

		outp( Base, Timer_CS+2 );
		wert |= (unsigned int)inp( Base+RD_DATA ) ;
		wert |= (unsigned int)inp( Base+RD_DATA )<< 8 ;

		timer_counter2=wert;
	}
	return wert;
};


int DAC98::TestCounterJMP( void )
{

	GateTimer( 0 ); 		
	GateCounter( 0 ); 		
	SetCounter( 0xffff );
	SetTimer( 0xffffffffUL ); 	
	GateTimer( 1 );		
	GateCounter (1);
	delay(1);
	GateTimer( 0 ); 		
	GateCounter( 0 ); 		
	int tim = (int) GetTimer();
	int cou = (int) GetCounter();
	int val=0;
	if (tim >= cou ) val=tim-cou; else val=cou-tim;
	if (val<20) return 1;
	CounterJMP = 1;
	return 0;
};



int DAC98::WaitCounter(double time)
{
	if (CounterJMP) return 0;
	if (time>16.0) {
		WaitCounter(time-16.0);
		time=16.0;}

	unsigned int SaveCounterSet=timer_counter2;
	int SaveCounterGate=CounterGate;

	GateCounter(0);
	SetCounter(0xFFFF);
	GateCounter(1);

	unsigned int ticks = int( 2.0 * 0.001 * time * Clock );
	unsigned int WaitEnd = 0xFFFF - ticks;
	while( GetCounter() > WaitEnd);


	GateCounter(0);	
	SetCounter(SaveCounterSet);
	GateCounter(SaveCounterGate);
	return 1;
}

void DAC98::GateCounter( int val )
{
	CounterGate=val;
	WriteDigital(9,val);
};

void DAC98::SetTimer ( unsigned long time )
{
	unsigned int w;

	/* Low Word setzen */
	w = ( unsigned int )time;
	timer_counter0=w;


	outp( Base, Timer_CS+3 );
	outp( Base+WR_DATA, 0x36 );

	outp( Base , Timer_CS+0 );
	outp( Base+WR_DATA, ( unsigned char)w );
	outp( Base+WR_DATA, ( unsigned char)(w>>8) );

	/* High Word setzen */
	w = ( unsigned int )(time>>16);
	timer_counter1=w;

	outp ( Base , Timer_CS+3 );
	outp ( Base+WR_DATA, 0x76 );

	outp( Base  ,  Timer_CS+1 );
	outp( Base+WR_DATA , ( unsigned char)w );
	outp( Base+WR_DATA , ( unsigned char)(w>>8) );
}

void DAC98::SetTimer( double time )
{
	unsigned long a, b;

	a = (unsigned long)sqrt( Clock /1.0e3 * time);
	b = a << 16;
	b |= (unsigned long)( Clock / 1.0e3 * time / (double)a);
	SetTimer(b);
};


unsigned long DAC98::GetTimer( void )
{
	unsigned long wert=0UL;

	outp( Base, Timer_CS+3 );
	outp( Base+WR_DATA, 0xe2);
	outp( Base, Timer_CS+0 );

	if ( inp( Base+RD_DATA ) & 64 ) wert |= timer_counter0;
	else
	{
		outp( Base, Timer_CS+3 );
		outp( Base+WR_DATA, 0xD2);
		outp( Base, Timer_CS+0 );
		wert |= (unsigned long)inp( Base+RD_DATA ) ;
		wert |= (unsigned long)inp( Base+RD_DATA )<< 8 ;

		timer_counter0=wert;
	}


	outp( Base, Timer_CS+3 );
	outp( Base+WR_DATA, 0xe4);

	outp( Base, Timer_CS+1 );

	if ( inp( Base+RD_DATA ) & 64 ) wert|=(timer_counter1<<16);
	else
	{
		outp( Base, Timer_CS+3 );
		outp( Base+WR_DATA, 0xD4);

		timer_counter1 = 0;
		outp( Base, Timer_CS+1 );
		timer_counter1 |= (unsigned long)inp( Base+RD_DATA );
		timer_counter1 |= (unsigned long)inp( Base+RD_DATA )<< 8 ;

		wert |= timer_counter1 << 16;
	}
	return wert;
};




void DAC98::GateTimer( int val )
{
	WriteDigital(8,val);
};


void DAC98::SetINT( int channel ,int val)
{
	outp(Base,Interrupt_CS);

	if( channel<0 || channel>4 ) return;

	int clock = intr_status & 0x70;
	intr_status = val ?
			 (intr_status &~(0x0001<<channel)) :
			 (intr_status | (0x0001<<channel));
	intr_status |= clock;
	outp(Base+WR_DATA,intr_status);
};

int DAC98::GetINT( void )
{
	outp(Base,Interrupt_CS);
	return inp( Base+RD_DATA);
};

int DAC98::GetINT( int channel )
{
	unsigned int mask;
	if((channel<0) || (channel<4)) return 0;
        mask = 0x0001<<channel;
	return( ( GetINT() & mask ) ? 0 : 1 );
};

void DAC98::ResetDDM( int channel )
{
	if( channel<0 || channel>2 ) return;
	outp( Base, ddm_adr[channel]);
	outp( Base+WR_DATA,0 );

};

void DAC98::ResetAllDDM( void )
{

	outp( Base, CSDDMALL_CS );
	outp( Base+WR_DATA,0 );

}

unsigned int DAC98::ReadDDM( int channel )
{
	outp( Base, ddm_adr[channel] +1);
	outp( Base+WR_DATA, 0);

	ddm_counter[channel] = 0;
	outp( Base, ddm_adr[channel] + DDMINCRHI);
	ddm_counter[channel] = inp( Base+RD_DATA ) << 8;

	outp( Base, ddm_adr[channel] + DDMINCRLO);
	ddm_counter[channel] |= inp( Base+RD_DATA );

	return( ddm_counter[channel] );
}

void DAC98::ReadAllDDM( void )
{
	// alle DDM'S umschalten
	outp( Base, CSDDMALL_CS +1 );
	outp( Base+WR_DATA, 0 );

	for( int f=0; f<3; f++){
	  ddm_counter[f] = 0;
	  outp( Base, ddm_adr[f] + DDMINCRHI);
	  ddm_counter[f] = inp( Base+RD_DATA ) << 8;

	  outp( Base, ddm_adr[f] + DDMINCRLO);
	  ddm_counter[f] |= inp( Base+RD_DATA );
	  }
}

int 	DAC98::ReadAnalogInt( int channel , int mode)
{
	int wert=0;

	outp ( Base , ADW_CS);	

	outp ( Base+WR_DATA , ( (mode<<3) | channel ) );
	while( ReadDigital( 8 ) );	// Busy

	outp ( Base , ADW_CS );	
	wert = inp ( Base+RD_DATA );		// auslesen Lowbyte

	outp ( Base , ADW_CS+1 );	
	wert |= inp ( Base+RD_DATA ) << 8;	// auslesen Highbyte

	return wert;
}
float  DAC98::ReadAnalogVolt( int channel , int mode)
{
	float wert;
	wert=(float)ReadAnalogInt(channel,mode);
	switch (mode) {
		case 0 :
			wert*=(5.0/4096.0);
			break;
		case 1 :
			wert*=(10.0/4096.0);
			break;
		case 2 :
			wert*=(10.0/4096.0);
			break;
		case 3 :
			wert*=(20.0/4096.0);
			break;
		default : break;
	}
	return wert;
}

void DAC98::WriteAnalogInt( int channel, int value )
{
	if( channel<0 || channel>1 ) return;
	if( channel ){
	  outp( Base, DAW1_CS );
	  aout1=value;
	  }
	else{
	  outp( Base, DAW0_CS );
	  aout0=value;
	  }
	outpw( Base+WR_DATA, value );
};

void DAC98::WriteAnalogVolt( int channel, float value )
{
	int v = (int)((value+10.0)/20.0*4096);
	if( v<0 ) v=0;
	if( v>4095 ) v=4095;
	WriteAnalogInt( channel, v );
};



DIC::DIC(int adr) : DAC98(adr)
{
	aout0	=	0;
	aout1	=	0;
	ident=0;
	ddm_counter[3] = 0;
};

DIC::~DIC()
{
};


void DIC::SetINT( int channel, int val )
{
	switch (channel)
	{
		case 4:
			DAC98::SetINT( 0, val );
			break;
		case 5:
			DAC98::SetINT( 1, val );
			break;
		default:
			break;
	}
};


void DIC::SetTimer( unsigned long time )
{
	if (time==0x12345678UL) ident=1;
	else ident=0;
	DAC98::SetTimer( time );
};



unsigned long DIC::GetTimer( void )
{
	unsigned long v = DAC98::GetTimer();

	if (ident && DAC98::Identifikation())
	{
		v = 0x12345678L;
		ident=0;
	}
	return v;

};

PCIO::PCIO(int adr) : DAC98(adr)
{
};

PCIO::~PCIO()
{
};

int PCIO::IsPCIO(void)
{
	return DAC98::Identifikation();
};

unsigned int PCIO::DigitalOutStatus(void)
{
	return (output_status_DAC98 & 0x00ff);
};

float PCIO::ReadAnalogVoltMean(int channel, int repeat)
{
	float 	mean;
	int 	count;

	mean = 0.0;
	for( count=0; count<repeat; count++ )
		mean += DAC98::ReadAnalogVolt( channel );
	mean /= (float)repeat;
	return( mean );
};

void PCIO::SetINT( int val )
{
	DAC98::SetINT( 0, val );
};

void PCIO::ResetHCTL(void)
{
	DAC98::ResetDDM(0);
};

int PCIO::ReadHCTL(void)
{
	return DAC98::ReadDDM(0);
};





